/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.chuck.service;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import io.netty.handler.codec.http.HttpHeaders;
import com.fasterxml.jackson.databind.ObjectMapper;

import org.apache.camel.AsyncCallback;
import org.apache.camel.Exchange;
import org.apache.camel.component.chuck.model.RandomJoke;
import org.apache.camel.support.GZIPHelper;
import org.apache.camel.util.IOHelper;
import org.asynchttpclient.AsyncHandler;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.HttpResponseBodyPart;
import org.asynchttpclient.HttpResponseStatus;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.asynchttpclient.util.HttpUtils.extractContentTypeCharsetAttribute;
import static org.asynchttpclient.util.MiscUtils.withDefault;

public class ApiServiceImpl implements ApiService {

    private static final Logger LOG = LoggerFactory.getLogger(ApiServiceImpl.class);

    private final AsyncHttpClient asyncHttpClient;
    private final ObjectMapper mapper;
    private final String baseUrl;
    private final int bufferSize;

    public ApiServiceImpl(AsyncHttpClient asyncHttpClient, int bufferSize, String baseUrl) {
        this.asyncHttpClient = asyncHttpClient;
        this.bufferSize = bufferSize;
        this.baseUrl = baseUrl;
        this.mapper = new ObjectMapper();
    }

    @Override
    public void randomJoke(Exchange exchange, AsyncCallback callback) {
        final String url = baseUrl + "/jokes/random";
        final RequestBuilder builder = new RequestBuilder("GET").setUrl(url);
        builder.setHeader("Content-Type", "application/json; charset=UTF-8");
        builder.setHeader("Accept", "application/json");
        asyncHttpClient.executeRequest(builder.build(), 
                new ChuckAsyncHandler(exchange, callback, url, bufferSize, mapper));
    }

    // this is not thread-safe and shouldn't be reused when executing concurrent requests
    private static final class ChuckAsyncHandler implements AsyncHandler<Exchange> {

        private final Exchange exchange;
        private final AsyncCallback callback;
        private final String url;
        private final ByteArrayOutputStream os;
        private final ObjectMapper mapper;
        private int statusCode;
        private String statusText;
        private String contentType;
        private String contentEncoding;
        private Charset charset;

        private ChuckAsyncHandler(Exchange exchange, AsyncCallback callback, String url, int bufferSize,
                ObjectMapper mapper) {
            this.exchange = exchange;
            this.callback = callback;
            this.url = url;
            this.os = new ByteArrayOutputStream(bufferSize);
            this.mapper = mapper;
        }

        @Override
        public void onThrowable(Throwable t) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} onThrowable {}", exchange.getExchangeId(), t);
            }
            exchange.setException(t);
            // processing completed by an async thread
            callback.done(false);
        }

        @Override
        public Exchange onCompleted() throws Exception {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} onCompleted", exchange.getExchangeId());
            }
            try {
                // copy from output stream to input stream
                os.flush();
                os.close();
                final boolean success = statusCode >= 200 && statusCode < 300;
                try (InputStream maybeGzStream = new ByteArrayInputStream(os.toByteArray());
                        InputStream is = GZIPHelper.uncompressGzip(contentEncoding, maybeGzStream);
                        Reader r = new InputStreamReader(is, charset)) {
                    if (success) {
                        final Object result;
                        if (LOG.isTraceEnabled()) {
                            final String body = IOHelper.toString(r);
                            LOG.trace("Received body for {}: {}", url, body);
                            result = mapper.readValue(body, RandomJoke.class);
                        } else {
                            result = mapper.readValue(r, RandomJoke.class);
                        }
                        exchange.getMessage().setBody(result);
                    } else {
                        throw new RuntimeException(
                                url + " responded: " + statusCode + " " + statusText + " " + IOHelper.toString(r));
                    }
                } catch (IOException e) {
                    throw new RuntimeException("Could not parse the response from " + url, e);
                }
            } catch (Exception e) {
                exchange.setException(e);
            } finally {
                // processing completed by an async thread
                callback.done(false);
            }
            return exchange;
        }

        @Override
        public String toString() {
            return "AhcAsyncHandler for exchangeId: " + exchange.getExchangeId() + " -> " + url;
        }

        @Override
        public State onBodyPartReceived(HttpResponseBodyPart bodyPart) throws Exception {
            // write body parts to stream, which we will bind to the Camel Exchange in onComplete
            os.write(bodyPart.getBodyPartBytes());
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} onBodyPartReceived {} bytes", exchange.getExchangeId(), bodyPart.length());
            }
            return State.CONTINUE;
        }

        @Override
        public State onStatusReceived(HttpResponseStatus responseStatus) throws Exception {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} onStatusReceived {}", exchange.getExchangeId(), responseStatus);
            }
            statusCode = responseStatus.getStatusCode();
            statusText = responseStatus.getStatusText();
            return State.CONTINUE;
        }

        @Override
        public State onHeadersReceived(HttpHeaders headers) throws Exception {
            contentEncoding = headers.get("Content-Encoding");
            contentType = headers.get("Content-Type");
            charset = withDefault(extractContentTypeCharsetAttribute(contentType), StandardCharsets.UTF_8);
            return State.CONTINUE;
        }
    }
    
}